import React, { useState, useCallback, useEffect, useReducer } from 'react';

import {
  LineUI,
  LineReducer,
  LineSetContext,
  Button,
} from 'scorer-ui-kit';

import Sidebar, { SidebarBox, SidebarHeading, BackLink, SidebarQuickLayout } from '../components/Sidebar';
import {Layout, Content, MainContainer} from '../components/Layout';


const Camareas: React.FC<{}> = () => {
  const [error, setError] = useState<string|null>('');
  const [showDelete, setShowDelete] = useState(false);
  const [state, dispatch] = useReducer(LineReducer, []);
  const HOSTNAME = window.location.hostname;
  const imgsrc = 'http://'+HOSTNAME+':21001/api/v1/stacks/RTSPcam1/snapshot?p='+(new Date()).getTime();
  const fetchAreas = useCallback(async () => {
    try {
      const response = await fetch('/api/camareas');
      if (!response.ok) {
        throw await response.json();
      }
      const {areas} = await response.json();

      dispatch({
        type: 'LOAD',
        state: areas
      });
    } catch (error) {
      setError('Failed to fetch areas');
    }
  },[]);

  const saveAreas = useCallback(async () => {
    try {
      const response = await fetch('/api/camareas', {
        method: 'PUT',
        body: JSON.stringify({
          areas: state
        }),
        headers: {
            'Content-Type': 'application/json'
        }
      });
      if (!response.ok) {
        throw await response.json();
      }

    } catch (error) {
      setError('Failed to save areas');
    }

  },[ state ]);

  const addArea = useCallback(async () => {
    dispatch({
      type: 'ADD_SET',
      data: {
        points: [
          {"x": 0, "y": 0},
          {"x": 500, "y": 0},
          {"x": 500, "y": 200},
          {"x": 0, "y": 200}
        ]
      }
    });
  },[]);

  const removeArea = useCallback(async (index) => {
    dispatch({
      type: 'REMOVE_SET',
      index
    });
  },[]);

  const addPoint = useCallback(async (index) => {
    dispatch({
      type: 'ADD_POINT',
      index
    });
  },[]);

  const removePoint = useCallback(async (index) => {
    if(state[index].points.length <= 3) return;
    dispatch({
      type: 'REMOVE_POINT',
      index
    });
  },[state]);

  useEffect(()=>{
    fetchAreas();
  }, [fetchAreas]);

  return (
    <Layout>
      <Sidebar>
        <SidebarBox>
          <BackLink to='/'>Back</BackLink>
        </SidebarBox>
        <SidebarBox>
          <SidebarHeading>Camera areas</SidebarHeading>
          <div>Area Name</div>
        </SidebarBox>

        {
          state.map((_shape, index) =>
            <SidebarBox key={index}>
              <SidebarQuickLayout direction='row'>
                <div>Area {index + 1}</div> {showDelete ?
                  <Button size='small' design='danger' onClick={()=>removeArea(index)}>Remove</Button>
                :
                  <>
                    <Button size='small' design='primary' onClick={()=>addPoint(index)}>+</Button>
                    <Button size='small' design='primary' onClick={()=>removePoint(index)}>-</Button>
                  </>}
              </SidebarQuickLayout>
            </SidebarBox>
          )
        }
        <SidebarBox>
          <Button size='small' design='primary' onClick={addArea}>Add Area</Button>
        </SidebarBox>
        <SidebarBox flex='1'>
          <SidebarQuickLayout direction='column'>
            <Button size='small' design='secondary' onClick={fetchAreas}>Cancel</Button>
            <Button size='small' design='primary' onClick={saveAreas}>Save</Button>
          </SidebarQuickLayout>
        </SidebarBox>
        <SidebarBox>
          <input type='checkbox' id='showDelete' name='showDelete' checked={showDelete} onClick={() => setShowDelete(!showDelete)} />
          <label htmlFor='showDelete'>Enable Area Removal</label>
        </SidebarBox>
      </Sidebar>
      <MainContainer>
        <Content>
          {error && <div>{error}</div>}
          <LineSetContext.Provider value={{state, dispatch}}>
            <LineUI  options={{showSetIndex: true}} src={imgsrc} />
          </LineSetContext.Provider>
        </Content>
      </MainContainer>
    </Layout>
  );
};

export default Camareas;
