import React, { useReducer, useCallback, useEffect, useState } from 'react';
import styled from 'styled-components';

import {
  LineUI,
  LineReducer,
  LineSetContext,
  Button,
  ButtonWithLoading
} from 'scorer-ui-kit';

import Sidebar, { SidebarBox, BackLink, SidebarQuickLayout } from '../components/Sidebar';
import {Layout, Content, MainContainer} from '../components/Layout';

const Heading = styled.div`
  font-size: 16px;
  font-weight: 500;
  color: hsl(208, 8%, 38%);
  margin: 10px 0;
`;
const Label = styled.span`
  display: inline-block;
  font-size: 14px;
  font-style: italic;
  line-height: 1.79;
  color: hsl(207, 5%, 57%);
  width: 55px;

`;

const Sketchlines: React.FC<{}> = () => {
  const [state, dispatch] = useReducer(LineReducer, []);
  const [error, setError] = useState<string|null>('');
  const [loading, setLoading] = useState(false);
  const [showDelete, setShowDelete] = useState(false);
  const HOSTNAME = window.location.hostname;
  const imgsrc = 'http://'+HOSTNAME+'/static/floor_plan.jpg?p='+(new Date()).getTime();
  const fetchLine = useCallback(async () => {
    try{
      const response = await fetch('/api/sketchlines');
      if (!response.ok) {
        throw await response.json();
      }
      const {lines} = await response.json();

      dispatch({
        type: 'LOAD',
        state: lines
      });
    } catch (error) {
      setError('Failed to fetch lines');
      //if (error.message) {
      //  setError(error.message);
      //} else {
      //  setError('Failed to fetch exclusions');
      //}
    }
  },[]);

  const addLine = useCallback(async () => {
    dispatch({
      type: 'ADD_SET',
      data: {
        points: [
          {x:0, y:0},
          {x:130, y:100},          
        ]
      }
    });
  }, []);

  const removeLine = useCallback(async (index) => {
    dispatch({
      type: 'REMOVE_SET',
      index
    });
  },[]);

  const saveLine = useCallback(async () => {
    setLoading(true);
    setError('');
    try {
      const response = await fetch('/api/sketchlines', {
        method: 'PUT',
        body: JSON.stringify({
            lines: state
        }),
        headers: {
            'Content-Type': 'application/json'
        }
      });
      if (!response.ok) {
        throw await response.json();
      }

    } catch (error) {
      setError('Failed to save lines');
      //if (error.message) {
      //  setError(error.message);
      //} else {
      //  setError('Failed to fetch exclusions');
      //}
    }
    setLoading(false);
  },[ state ]);

  useEffect(()=>{
    fetchLine();
  },[fetchLine]);

  return (
    <Layout>
      <Sidebar>
        <SidebarBox>
          <BackLink to='/'>Back</BackLink>
          <Heading>Setup Sketch Lines</Heading>
        </SidebarBox>

        {
          state.map((_shape, index) =>
            <SidebarBox key={index}>
              <SidebarQuickLayout direction='row'>
                <Heading>Line {index}</Heading> {showDelete ?
                  <Button size='small' design='danger' onClick={()=>removeLine(index)}>Remove</Button>
                :
                  <>
                  </>}
              </SidebarQuickLayout>

              <div>Line Start</div>
              <Label>X: {_shape.points[0].x}</Label>
              <Label>Y: {_shape.points[0].y}</Label>
              <div>Line End</div>
              <Label>X: {_shape.points[1].x}</Label>
              <Label>Y: {_shape.points[1].y}</Label>

            </SidebarBox>
          )
        }

        <SidebarBox>
          <Button size='small' design='primary' onClick={addLine}>Add Line</Button>
        </SidebarBox>

        <SidebarBox>
          <input type='checkbox' id='showDelete' name='showDelete' checked={showDelete} onClick={() => setShowDelete(!showDelete)} />
          <label htmlFor='showDelete'>Enable Line Removal</label>
        </SidebarBox>

        <SidebarBox flex='1'>
          <SidebarQuickLayout direction='column'>
            <Button size='small' design='secondary' onClick={fetchLine}>Cancel</Button>
            <ButtonWithLoading loading={loading} size='small' design='primary' onClick={saveLine}>Save</ButtonWithLoading>
          </SidebarQuickLayout>
        </SidebarBox>
      </Sidebar>
      <MainContainer>
        <LineSetContext.Provider value={{state, dispatch}}>
          <Content>
            {error && <div>{error}</div>}
            <LineUI  options={{showSetIndex: true}} src={imgsrc} />
          </Content>
        </LineSetContext.Provider>
      </MainContainer>
    </Layout>
  );
};

export default Sketchlines;
