#!/usr/bin/env python3

# This script dumps various streams

import sys
import os
import argparse
import pickle
import json
import zmq
import numpy
import cv2
import math
from datetime import datetime, timedelta, timezone


# Handle arguments

PROG_DESCRIPTION='''\
description:
  this program subscribes ZMQ streams and dumps the data to stdout
'''

PROG_EPILOG='''\

examples:
  %(prog)s
  %(prog)s --pretty-json
  %(prog)s -ic tcp://192.168.10.11:9877 tcp://192.168.10.12:9877
'''

DEFAULT_INGRESS_ADDR="tcp://localhost:5002"
DEFAULT_INGRESS_TOPICS=[ 'VideoFrame', 'JpegFrame', 'LogFrame' ]


ap = argparse.ArgumentParser(
        formatter_class=argparse.RawDescriptionHelpFormatter,
        description=PROG_DESCRIPTION,
        epilog=PROG_EPILOG)

ap.add_argument('-v','--verbose', action='count', default=0, help='increase verbosity')

apg1 = ap.add_argument_group()
apmx1 = apg1.add_mutually_exclusive_group()
apmx1.add_argument('-ic', dest='ingress_connects', metavar='ADDR', nargs='+', action='append',
                   help='specify a ZMQ endpoint to connect to a data source (default:"{}")'
                       .format(DEFAULT_INGRESS_ADDR))

apg3 = ap.add_argument_group()
apg3.add_argument('--subscribe', dest='topics', metavar='TOPIC', nargs='+', action='append',
                  help='specify a subscription topic of data sources (default:"{}")'
                       .format(DEFAULT_INGRESS_TOPICS))
apg3.add_argument('--pretty-json', action='store_true',
                  help='dump the json with sorted keys and indent')

args = ap.parse_args()


# Flatten the lists
if args.ingress_connects:
    args.ingress_connects = sum(args.ingress_connects, [])
    args.ingress_connects = [s for s in args.ingress_connects if s]
if args.topics:
    args.topics = sum(args.topics, [])
    args.topics = [s for s in args.topics if s]

# Set the default endpoint
if not args.ingress_connects:
    args.ingress_connects = [ DEFAULT_INGRESS_ADDR ]

# Set the default topic filter
if not args.topics:
    args.topics = DEFAULT_INGRESS_TOPICS

# Set the option for json.dumps
if args.pretty_json:
    json_opts = {'sort_keys': True, 'indent': 2}
else:
    json_opts = {'sort_keys': True}



# --------------------------------
# Utility Function
# --------------------------------

def dprint(level_, *args_, **kwargs_):
    if level_ <= 0:
        print('Log level must be positive: {}'.format(level_))
        sys.exit(1)

    # args.verbose:
    #    0: Suppress all debug logging
    #    1: Show significant logs only
    #    2: Show important logs
    #    3: Show detailed logs
    #    4: Show trace logs
    if args.verbose >= level_:
        print(*args_, **kwargs_)



# --------------------------------
# Stream Subscriber
# --------------------------------
TOPIC_LOG_FRAME = b'LogFrame'
TOPIC_VIDEO_FRAME = b'VideoFrame'
TOPIC_JPEG_FRAME = b'JpegFrame'
POLL_WAIT_TIME  = 30           # milliseconds


def subscribe_streams(ctx_, stream_addrs_):
    print('Starting the stream subscriber now...\n')
    
    dirpath = 'data'
    
    # Setup the ingress socket for streams
    socki = ctx_.socket(zmq.SUB)
    socki.setsockopt(zmq.RCVHWM, 100)
    if type(args.topics) == list:
        for topic in args.topics:
            print('Ingress: Setting a topic filter "{}"'.format(topic))
            socki.setsockopt_string(zmq.SUBSCRIBE, topic)

    for addr in stream_addrs_:
        print('Connecting to "{}"'.format(addr))
        socki.connect(addr)

    poller = zmq.Poller()
    poller.register(socki, zmq.POLLIN)

    print('\nReceiving the stream data ...\n')
    JST = timezone(timedelta(hours=+9), 'JST')
    jsonpack = []
    datestr0 = datetime.now(JST).strftime('%Y%m%d_%H%M%S')
    datenum0 = datetime.now(JST)
    
    if not os.path.exists(dirpath+'/'+datestr0[:8]):
        os.makedirs(dirpath+'/'+datestr0[:8])
    
    raw_json_file = open(dirpath+'/'+datestr0[:8]+'/raw_'+datestr0+'.json', 'w')

        
    while True:

        events = dict(poller.poll(POLL_WAIT_TIME))

        if events.get(socki) != zmq.POLLIN:
            continue    # Poller time out


        # Receive a message
        msg = socki.recv_multipart(flags=zmq.NOBLOCK)
        if not msg:
            continue

        # Decode the message
        try:
            topic       = msg[0]  # bytes
            stream_id   = msg[1]  # bytes
            frame_time  = pickle.loads(msg[2])

            dprint(4,'Received: topic {} stream_id {} ftime {:.3f}'
                     .format(topic, stream_id, frame_time), flush=True)

            if topic.endswith(b'/' + stream_id):
                # Separate out the stream ID from the topic
                topic = topic[:-(len(stream_id)+1)]

            if topic == TOPIC_VIDEO_FRAME or topic == TOPIC_JPEG_FRAME:
                meta        = pickle.loads(msg[3])
                img         = numpy.frombuffer(memoryview(msg[4]), dtype=meta['dtype'])
                img         = img.reshape(meta['shape'])
                
                if topic == TOPIC_JPEG_FRAME:
                    img = cv2.imdecode(img, cv2.IMREAD_COLOR)
                    #cv2.imwrite("test.jpg",img)
                item        = (img, pickle.loads(msg[5]))

            elif topic == TOPIC_LOG_FRAME:
                item        = (numpy.array([]), pickle.loads(msg[3]))
            else:
                dprint(4,'Ignoring a message by topic: {} (len {})'
                          .format(topic, len(msg)))
                continue

        except pickle.UnpicklingError as e:
            print('Corrupted pickle message: topic {}, stream {}, {}'
                  .format(topic, stream_id, e))
            continue
        except IndexError as e:
            print('Invalid length: topic {}, stream {}, length {}, {}'
                  .format(topic, stream_id, len(msg)))
            continue
        except ValueError as e:
            print('Invalid value: topic {}, stream {}, {}'
                  .format(topic, stream_id, e))
            continue

        # Dump the message
        item[1]['stream_id'] = stream_id.decode()
        item[1]['frame_time'] = frame_time
        
        jptime = datetime.fromtimestamp(item[1]['frame_time'],JST)
        print(jptime)
        jsonpack.append(item[1])
        datenum1 = datetime.now(JST)
        datestr1 = datetime.now(JST).strftime('%Y%m%d_%H%M%S')
        datenumdiff = datenum1-datenum0
        if datenumdiff.total_seconds()>60:
            datenum0=datenum1
            datestr0=datestr1
            json.dump(jsonpack,raw_json_file)
            
            if not os.path.exists(dirpath+'/'+datestr0[:8]):
                os.makedirs(dirpath+'/'+datestr0[:8])

            raw_json_file = open(dirpath+'/'+datestr0[:8]+'/raw_'+datestr0+'.json', 'w')
            jsonpack = []

    # Clean up
    socki.setsockopt(zmq.LINGER, 0)
    socki.close()
    dprint(4,'Exiting the stream subscriber.')



# Main
ctx = zmq.Context()

try:
    subscribe_streams(ctx, args.ingress_connects)

except KeyboardInterrupt:
    
    print("\nKeyboardInterrupt\n", file=sys.stderr, flush=True)

ctx.term()

# EOF